
# Public IP for VyOS VM
resource "azurerm_public_ip" "vyos_vm_public_ip" {
  name                    = "vyov-vm-public-ip"
  location                = var.location
  resource_group_name     = var.resource_group
  allocation_method       = "Static"
  sku                     = "Standard"
  tags                    = var.tags
  idle_timeout_in_minutes = "30"

  timeouts {
    create = "15m"
    delete = "10m"
  }
}

# Public NIC for WAN
resource "azurerm_network_interface" "vyos_pub_nic" {
  name                = "vyos-pub-nic"
  location            = var.location
  resource_group_name = var.resource_group
  tags                = var.tags

  ip_configuration {
    name                          = "public"
    subnet_id                     = azurerm_subnet.test_vnet_public.id
    private_ip_address_allocation = "Dynamic"
    public_ip_address_id          = azurerm_public_ip.vyos_vm_public_ip.id
  }

  timeouts {
    create = "15m"
    delete = "10m"
  }
}

# Private NICs for internal communication
resource "azurerm_network_interface" "vyos_priv_nic" {
  count               = 3 # Adjust count for the number of private NICs needed
  name                = "vyos-priv-nic-${count.index + 1}"
  location            = var.location
  resource_group_name = var.resource_group
  tags                = var.tags

  ip_forwarding_enabled          = true
  accelerated_networking_enabled = true

  ip_configuration {
    name                          = "private-${count.index + 1}"
    subnet_id                     = azurerm_subnet.test_vnet_private.id
    private_ip_address_allocation = "Dynamic"
  }

  timeouts {
    create = "25m"
    delete = "20m"
  }
}

# VyOS virtual machine parameters
resource "azurerm_linux_virtual_machine" "vyos_vm" {
  name                = "vyos-vm"
  location            = var.location
  resource_group_name = var.resource_group
  size                = var.vyos_vm_size
  tags                = var.tags

  network_interface_ids = concat(
    [azurerm_network_interface.vyos_pub_nic.id],                  # Public NIC
    [for nic in azurerm_network_interface.vyos_priv_nic : nic.id] # Dynamically attach all private NICs
  )

  admin_username                  = var.admin_username
  admin_password                  = var.admin_password
  disable_password_authentication = false
  custom_data = base64encode(templatefile("${path.module}/files/vyos_user_data.tpl", {
  }))

  os_disk {
    name                 = "vyos-vm-osdisk"
    caching              = "ReadWrite"
    storage_account_type = "Standard_LRS"
  }

   source_image_reference {
     publisher = var.image_publisher
     offer     = var.image_offer
     sku       = var.image_sku
     version   = var.image_version
   }

   plan {
     name      = var.image_sku
     publisher = var.image_publisher
     product   = var.image_offer
   }

  timeouts {
    create = "90m"
    delete = "30m"
  }
}
